//pattern: 'presence',

const { Module } = require('../main');
const config = require('../config');

let presenceIntervals = {};

function savePresenceSettings() {
    const activePresences = {};
    Object.keys(presenceIntervals).forEach(jid => {
        activePresences[jid] = presenceIntervals[jid].type;
    });
    config.ACTIVE_PRESENCES = JSON.stringify(activePresences);
}

async function loadPresenceSettings(client) {
    try {
        if (config.ACTIVE_PRESENCES) {
            const activePresences = JSON.parse(config.ACTIVE_PRESENCES);
            for (const [jid, type] of Object.entries(activePresences)) {
                await startPresence(client, jid, type);
            }
        }
    } catch (error) {
        console.error('Failed to load presence settings:', error);
    }
}

async function startPresence(client, jid, type) {
    if (presenceIntervals[jid]) {
        clearInterval(presenceIntervals[jid].interval);
    }
    
    const interval = setInterval(async () => {
        try {
            await client.sendPresenceUpdate(type, jid);
        } catch (e) {}
    }, 10000);
    
    presenceIntervals[jid] = {
        interval: interval,
        type: type
    };
    
    try {
        await client.sendPresenceUpdate(type, jid);
    } catch (e) {}
    
    savePresenceSettings();
}

function stopPresence(jid) {
    if (presenceIntervals[jid]) {
        clearInterval(presenceIntervals[jid].interval);
        delete presenceIntervals[jid];
        savePresenceSettings();
    }
}

Module({
    on: 'start'
}, async (client) => {
    setTimeout(async () => {
        await loadPresenceSettings(client);
    }, 3000);
});

Module({
    pattern: 'presence ?(.*)',
    fromMe: true,
    desc: 'Set typing/recording presence 24/7',
    type: 'utility'
}, async (message, match) => {
    let input = match[1];
    
    if (!input) {
        return await message.sendReply(`*Presence Control Menu*
• \`presence typing\` - Start typing here
• \`presence recording\` - Start recording here  
• \`presence typing jid\` - Start typing in specific chat
• \`presence stop\` - Stop presence here
• \`presence list\` - Show active presences
• \`presence stopall\` - Stop all presences`);
    }
    
    let args = input.split(' ');
    let cmd = args[0].toLowerCase();
    
    if (cmd === 'typing' || cmd === 'recording') {
        let targetJid = args[1] || message.jid;
        let presenceType = cmd === 'typing' ? 'composing' : 'recording';
        
        await startPresence(message.client, targetJid, presenceType);
        await message.sendReply(`✅ Started ${cmd} presence`);
        
    } else if (cmd === 'stop') {
        let targetJid = args[1] || message.jid;
        stopPresence(targetJid);
        try {
            await message.client.sendPresenceUpdate('available', targetJid);
        } catch (e) {}
        await message.sendReply(`✅ Stopped presence`);
        
    } else if (cmd === 'list') {
        let active = Object.keys(presenceIntervals);
        if (active.length === 0) {
            await message.sendReply('No active presences');
        } else {
            let list = active.map((jid, i) => {
                const type = presenceIntervals[jid].type;
                return `${i+1}. ${jid} (${type})`;
            }).join('\n');
            await message.sendReply(`*Active Presences:*\n${list}`);
        }
        
    } else if (cmd === 'stopall') {
        Object.keys(presenceIntervals).forEach(jid => {
            stopPresence(jid);
            try {
                message.client.sendPresenceUpdate('available', jid);
            } catch (e) {}
        });
        await message.sendReply('✅ Stopped all presences');
        
    } else {
        await message.sendReply('Invalid command. Use: typing, recording, stop, list, or stopall');
    }
});

Module({
    pattern: 'typing ?(.*)',
    fromMe: true,
    desc: 'Start typing presence',
    type: 'utility'
}, async (message, match) => {
    let jid = match[1] || message.jid;
    await startPresence(message.client, jid, 'composing');
    await message.sendReply('✅ Typing started');
});

Module({
    pattern: 'recording ?(.*)',
    fromMe: true,
    desc: 'Start recording presence',
    type: 'utility'
}, async (message, match) => {
    let jid = match[1] || message.jid;
    await startPresence(message.client, jid, 'recording');
    await message.sendReply('✅ Recording started');
});

Module({
    pattern: 'stoppresence ?(.*)',
    fromMe: true,
    desc: 'Stop presence',
    type: 'utility'
}, async (message, match) => {
    let jid = match[1] || message.jid;
    stopPresence(jid);
    try {
        await message.client.sendPresenceUpdate('available', jid);
    } catch (e) {}
    await message.sendReply('✅ Presence stopped');
});