const { Module } = require('../main');
const { DataTypes } = require('sequelize');
const { sequelize } = require('../config');
const axios = require('axios');

const GroupSnapshot = sequelize.define('GroupSnapshot', {
    groupId: {
        type: DataTypes.STRING,
        primaryKey: true,
        allowNull: false
    },
    groupName: {
        type: DataTypes.TEXT,
        allowNull: true
    },
    groupDescription: {
        type: DataTypes.TEXT,
        allowNull: true
    },
    groupIcon: {
        type: DataTypes.TEXT,
        allowNull: true
    },
    createdAt: {
        type: DataTypes.DATE,
        defaultValue: DataTypes.NOW
    }
});

GroupSnapshot.sync();

Module({
    pattern: 'snapshot ?(.*)',
    fromMe: true,
    desc: 'Manage group snapshots (create, restore, delete)',
    type: 'group'
}, async (message, match) => {
    if (!message.isGroup) {
        return await message.sendReply('_This command can only be used in groups_');
    }

    const action = match[1] ? match[1].toLowerCase().trim() : '';

    try {
        if (action === 'create') {
            const groupMetadata = await message.client.groupMetadata(message.jid);
            const groupName = groupMetadata.subject || '';
            const groupDescription = groupMetadata.desc || '';

            let groupIconBase64 = null;
            try {
                const groupPicture = await message.client.profilePictureUrl(message.jid, 'image');
                const response = await axios.get(groupPicture, { responseType: 'arraybuffer' });
                const imageBuffer = Buffer.from(response.data);
                groupIconBase64 = imageBuffer.toString('base64');
            } catch (error) {
                groupIconBase64 = null;
            }

            await GroupSnapshot.upsert({
                groupId: message.jid,
                groupName: groupName,
                groupDescription: groupDescription,
                groupIcon: groupIconBase64
            });

            await message.sendReply(`_*Group Snapshot Created Successfully*_\n\n_*Captured Details:*_\n_• Name: ${groupName || 'No name set'}_\n_• Description: ${groupDescription ? 'Present' : 'No description set'}_\n_• Icon: ${groupIconBase64 ? 'Present' : 'No icon set'}_\n\n_This snapshot can be restored later if group settings are changed._`);

        } else if (action === 'restore') {
            const groupMetadata = await message.client.groupMetadata(message.jid);
            const participants = groupMetadata.participants;
            const botNumber = message.client.user.id.split(':')[0] + '@s.whatsapp.net';
            const botParticipant = participants.find(p => p.id === botNumber);

            if (!botParticipant || !botParticipant.admin) {
                return await message.sendReply('_I need admin privileges to restore group settings_');
            }

            const snapshot = await GroupSnapshot.findOne({
                where: { groupId: message.jid }
            });

            if (!snapshot) {
                return await message.sendReply('_No snapshot found for this group. Create one first using .snapshot create_');
            }

            let restoredItems = [];
            let failedItems = [];

            if (snapshot.groupName && snapshot.groupName.trim() !== '') {
                try {
                    await message.client.groupUpdateSubject(message.jid, snapshot.groupName);
                    restoredItems.push('Group name');
                } catch (error) {
                    failedItems.push('Group name');
                }
            }

            if (snapshot.groupDescription && snapshot.groupDescription.trim() !== '') {
                try {
                    await message.client.groupUpdateDescription(message.jid, snapshot.groupDescription);
                    restoredItems.push('Group description');
                } catch (error) {
                    failedItems.push('Group description');
                }
            }

            if (snapshot.groupIcon) {
                try {
                    const imageBuffer = Buffer.from(snapshot.groupIcon, 'base64');
                    await message.client.setProfilePicture(message.jid, imageBuffer);
                    restoredItems.push('Group icon');
                } catch (error) {
                    failedItems.push('Group icon');
                }
            }

            let reportMessage = '_*Group Restoration Complete*_\n\n';

            if (restoredItems.length > 0) {
                reportMessage += '_*Successfully Restored:*_\n';
                restoredItems.forEach(item => {
                    reportMessage += `_• ${item}_\n`;
                });
                reportMessage += '\n';
            }

            if (failedItems.length > 0) {
                reportMessage += '_*Failed to Restore:*_\n';
                failedItems.forEach(item => {
                    reportMessage += `_• ${item}_\n`;
                });
                reportMessage += '\n';
            }

            if (restoredItems.length === 0 && failedItems.length === 0) {
                reportMessage += '_No changes were needed - group already matches snapshot._';
            }

            await message.sendReply(reportMessage.trim());

        } else if (action === 'delete') {
            const deleted = await GroupSnapshot.destroy({
                where: { groupId: message.jid }
            });

            if (deleted) {
                await message.sendReply('_Group snapshot deleted successfully_');
            } else {
                await message.sendReply('_No snapshot found for this group_');
            }

        } else {
            const snapshot = await GroupSnapshot.findOne({
                where: { groupId: message.jid }
            });

            const hasSnapshot = snapshot ? true : false;
            const createdDate = snapshot ? new Date(snapshot.createdAt).toLocaleString() : '';

            let menuMessage = `_*Group Snapshot Manager*_\n\n_*Available Commands:*_\n_• .snapshot create - Create a new snapshot_\n_• .snapshot restore - Restore from snapshot_\n_• .snapshot delete - Delete current snapshot_\n\n_*Current Status:* ${hasSnapshot ? `Snapshot exists (Created: ${createdDate})` : 'No snapshot found'}_`;

            await message.sendReply(menuMessage.trim());
        }

    } catch (error) {
        console.error('Error in snapshot command:', error);
        await message.sendReply('_An error occurred while processing the command_');
    }
}); 