const { Module } = require('../main');
const { MODE } = require('../config');
const axios = require('axios');
const { exec } = require('child_process');

const isPrivateMode = MODE === 'private';

function formatBytes(bytes, decimals = 2) {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const dm = decimals < 0 ? 0 : decimals;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(dm)) + ' ' + sizes[i];
}

function formatSpeed(bytesPerSecond) {
    const mbps = (bytesPerSecond * 8) / (1024 * 1024);
    return mbps.toFixed(2) + ' Mbps';
}

async function testDownloadSpeed() {
    const testUrl = 'https://proof.ovh.net/files/10Mb.dat';
    try {
        const startTime = performance.now();
        const response = await axios.get(testUrl, {
            responseType: 'arraybuffer',
            timeout: 30000
        });
        const endTime = performance.now();
        const duration = (endTime - startTime) / 1000;
        const bytes = response.data.byteLength;
        return {
            success: true,
            bytes: bytes,
            duration: duration,
            speed: bytes / duration
        };
    } catch (error) {
        return { success: false, error: error.message };
    }
}

async function testUploadSpeed() {
    const testData = Buffer.alloc(2 * 1024 * 1024);
    try {
        const startTime = performance.now();
        await axios.post('https://postman-echo.com/post', testData, {
            headers: { 'Content-Type': 'application/octet-stream' },
            timeout: 30000
        });
        const endTime = performance.now();
        const duration = (endTime - startTime) / 1000;
        return {
            success: true,
            bytes: testData.length,
            duration: duration,
            speed: testData.length / duration
        };
    } catch (error) {
        return { success: false, error: error.message };
    }
}

async function getServerInfo() {
    try {
        const response = await axios.get('https://ipapi.co/json/', { timeout: 10000 });
        return {
            success: true,
            ip: response.data.ip,
            city: response.data.city,
            country: response.data.country_name,
            isp: response.data.org
        };
    } catch (error) {
        return { success: false, error: error.message };
    }
}

const runLegacyTest = async (message, startMsg) => {
    try {
        const serverInfo = await getServerInfo();
        let statusMsg = '`Network Speed Test (Legacy Fallback)`\n\n';
        if (serverInfo.success) {
            statusMsg += `- ISP: \`${serverInfo.isp}\`\n`;
            statusMsg += `- Location: \`${serverInfo.city}, ${serverInfo.country}\`\n\n`;
        }

        statusMsg += '`Testing Download Speed...`';
        await message.edit(statusMsg, message.jid, startMsg.key);
        const downloadResult = await testDownloadSpeed();
        if (downloadResult.success) {
            statusMsg += ` ✓\n- Download: \`${formatSpeed(downloadResult.speed)}\`\n\n`;
        } else {
            statusMsg += ` ✗\n- Error: \`${downloadResult.error}\`\n\n`;
        }

        statusMsg += '`Testing Upload Speed...`';
        await message.edit(statusMsg, message.jid, startMsg.key);
        const uploadResult = await testUploadSpeed();
        if (uploadResult.success) {
            statusMsg += ` ✓\n- Upload: \`${formatSpeed(uploadResult.speed)}\`\n`;
        } else {
            statusMsg += ` ✗\n- Error: \`${uploadResult.error}\`\n`;
        }
        
        await message.edit(statusMsg, message.jid, startMsg.key);
    } catch (error) {
        await message.edit(`\`Speed Test Failed\`\n\n- Error: \`${error.message}\``, message.jid, startMsg.key);
    }
};

Module({
    pattern: 'speedtest',
    fromMe: isPrivateMode,
    desc: 'Test network speed (download/upload)',
    use: 'utility'
}, async (message, match) => {
    const startMsg = await message.sendReply('_Running Speed Test..._\n\n_This may take a few moments._');
    const command = 'curl -s https://raw.githubusercontent.com/sivel/speedtest-cli/master/speedtest.py | python3 -';

    exec(command, async (error, stdout, stderr) => {
        if (error) {
            await message.edit('_Python method failed. Using legacy fallback..._', message.jid, startMsg.key);
            await runLegacyTest(message, startMsg);
            return;
        }

        try {
            const serverInfo = await getServerInfo();
            let resultText = '`Network Speed Test`\n\n';

            if (serverInfo.success) {
                resultText += `- IP: \`${serverInfo.ip}\`\n`;
                resultText += `- ISP: \`${serverInfo.isp}\`\n`;
            }

            const downloadMatch = stdout.match(/Download: ([\d.]+\s+Mbit\/s)/);
            const uploadMatch = stdout.match(/Upload: ([\d.]+\s+Mbit\/s)/);
            const serverMatch = stdout.match(/Hosted by (.*?) \[(.*?)\]: ([\d.]+\s+ms)/);

            if (serverMatch) {
                resultText += `- Test Server: \`${serverMatch[1].trim()}\`\n`;
                resultText += `- Ping: \`${serverMatch[3]}\`\n`;
            }
            if (downloadMatch) {
                resultText += `- Download: \`${downloadMatch[1]}\`\n`;
            }
            if (uploadMatch) {
                resultText += `- Upload: \`${uploadMatch[1]}\`\n`;
            }

            if (!downloadMatch || !uploadMatch || !serverMatch) {
                 throw new Error('Could not parse speedtest-cli output.');
            }

            await message.edit(resultText, message.jid, startMsg.key);
        } catch (e) {
            await message.edit(`\`Speed Test Parse Failed\`\n\n- Error: \`${e.message}\`\n\nFalling back...`, message.jid, startMsg.key);
            await runLegacyTest(message, startMsg);
        }
    });
});