const { Module } = require('../main');
const fs = require('fs');
const path = require('path');
const axios = require('axios'); 

const WORD_LIST_URL = 'https://rawcdn.githack.com/dwyl/english-words/20f5cc9b3f0ccc8ce45d814c532b7c2031bba31c/words_alpha.txt';
let WORD_SET;

async function loadWordsFromUrl() {
  try {
    const response = await axios.get(WORD_LIST_URL);
    const words = response.data.split('\n')
      .map(word => word.trim().toLowerCase())
      .filter(word => word.length >= 3);
    return new Set(words);
  } catch (error) {
    console.error('Error loading word list from URL:', error);
    return null;
  }
}

(async function initWordSet() {
  try {
    WORD_SET = await loadWordsFromUrl();
    if (WORD_SET) {
      console.log(`Loaded ${WORD_SET.size} words from URL`);
    } else {
      throw new Error('Failed to load from URL');
    }
  } catch (error) {
    console.error('Error initializing word set:', error);

    WORD_SET = new Set([
      'yes', 'you', 'your', 'year', 'young', 'yellow', 'yard', 'yell', 'yolk', 'yarn',
      'key', 'keep', 'kind', 'know', 'king', 'kick', 'kill', 'kiss', 'knee', 'knife',

    ]);
    console.log('Using fallback word list');
  }
})();

const activeGames = new Map();
const gameTimers = new Map();

function getDynamicDifficulty(totalWords, activePlayerCount = 2) {

  let thresholds = {
    easy: 8,
    medium: 20,
    challenging: 40,
    hard: 65,
    expert: 95,
    master: 130
  };

  if (activePlayerCount <= 2) {

    thresholds = {
      easy: 5,
      medium: 12,
      challenging: 24,
      hard: 40,
      expert: 60,
      master: 80
    };
  } else if (activePlayerCount === 3) {

    thresholds = {
      easy: 6,
      medium: 15,
      challenging: 30,
      hard: 50,
      expert: 75,
      master: 100
    };
  } else if (activePlayerCount === 4) {

    thresholds = {
      easy: 7,
      medium: 17,
      challenging: 35,
      hard: 55,
      expert: 80,
      master: 110
    };
  }

  if (totalWords < thresholds.easy) {
    return { minLength: 3, timeLimit: 40000, level: 'Beginner' };
  } else if (totalWords < thresholds.medium) {
    return { minLength: 4, timeLimit: 35000, level: 'Easy' };
  } else if (totalWords < thresholds.challenging) {
    return { minLength: 4, timeLimit: 30000, level: 'Medium' };
  } else if (totalWords < thresholds.hard) {
    return { minLength: 5, timeLimit: 25000, level: 'Challenging' };
  } else if (totalWords < thresholds.expert) {
    return { minLength: 5, timeLimit: 20000, level: 'Hard' };
  } else if (totalWords < thresholds.master) {
    return { minLength: 6, timeLimit: 18000, level: 'Expert' };
  } else {
    return { minLength: 6, timeLimit: 15000, level: 'Master' };
  }
}

class WordChainGame {
  constructor(jid) {
    this.jid = jid;
    this.players = [];
    this.currentPlayerIndex = 0;
    this.gameState = 'waiting'; 
    this.currentLetter = this.getRandomLetter();
    this.usedWords = new Set();
    this.totalWords = 0;
    this.longestWord = '';
    this.longestWordPlayer = '';
    this.startTime = null;
    this.joinTimer = null;
    this.turnTimer = null;
    this.currentDifficulty = getDynamicDifficulty(0, 2); 
    this.processingWord = false; 
    this.currentTurnStartTime = null; 
    this.turnNumber = 0; 
  }

  getRandomLetter() {
    const letters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
    return letters[Math.floor(Math.random() * letters.length)];
  }

  addPlayer(userId, username) {
    if (!this.players.find(p => p.id === userId)) {
      this.players.push({ id: userId, username, words: 0, eliminated: false });
      return true;
    }
    return false;
  }

  getCurrentPlayer() {
    const activePlayers = this.players.filter(p => !p.eliminated);
    if (activePlayers.length === 0) return null;
    return activePlayers[this.currentPlayerIndex % activePlayers.length];
  }

  getNextPlayer() {
    const activePlayers = this.players.filter(p => !p.eliminated);
    if (activePlayers.length === 0) return null;
    const nextIndex = (this.currentPlayerIndex + 1) % activePlayers.length;
    return activePlayers[nextIndex];
  }

  nextTurn() {
    const activePlayers = this.players.filter(p => !p.eliminated);
    this.currentPlayerIndex = (this.currentPlayerIndex + 1) % activePlayers.length;
    this.turnNumber++; 
    this.currentTurnStartTime = Date.now(); 
  }

  eliminatePlayer(playerId) {
    const player = this.players.find(p => p.id === playerId);
    if (player && !player.eliminated) {
      player.eliminated = true;

      const activePlayers = this.players.filter(p => !p.eliminated);
      if (activePlayers.length > 0) {

        const activeIndex = activePlayers.findIndex(p => p.id === playerId);
        if (activeIndex !== -1 && activeIndex <= this.currentPlayerIndex) {
          this.currentPlayerIndex = Math.max(0, this.currentPlayerIndex - 1);
        }

        if (this.currentPlayerIndex >= activePlayers.length) {
          this.currentPlayerIndex = 0;
        }
      }

      return true;
    }
    return false;
  }

  updateDifficulty() {
    const activePlayers = this.getActivePlayers();
    const oldLevel = this.currentDifficulty.level;
    this.currentDifficulty = getDynamicDifficulty(this.totalWords, activePlayers.length);
    const levelChanged = oldLevel !== this.currentDifficulty.level;

    console.log(`Difficulty Update: Words=${this.totalWords}, Players=${activePlayers.length}, Old=${oldLevel}, New=${this.currentDifficulty.level}, Changed=${levelChanged}`);

    return levelChanged;
  }

  isValidWord(word) {
    return WORD_SET.has(word.toLowerCase());
  }

  getActivePlayers() {
    return this.players.filter(p => !p.eliminated);
  }

  getGameDuration() {
    if (!this.startTime) return '00:00:00';
    const duration = Date.now() - this.startTime;
    const seconds = Math.floor(duration / 1000);
    const minutes = Math.floor(seconds / 60);
    const hours = Math.floor(minutes / 60);
    return `${hours.toString().padStart(2, '0')}:${(minutes % 60).toString().padStart(2, '0')}:${(seconds % 60).toString().padStart(2, '0')}`;
  }

  getDifficultyEmoji() {
    switch(this.currentDifficulty.level) {
      case 'Beginner': return '🟢';
      case 'Easy': return '🟡';
      case 'Medium': return '🟠';
      case 'Challenging': return '🔴';
      case 'Hard': return '🟤';
      case 'Expert': return '🟣';
      case 'Master': return '⚫';
      default: return '🔵';
    }
  }

  getDifficultyProgressInfo() {
    const activePlayers = this.getActivePlayers().length;
    let nextThreshold;

    if (activePlayers <= 2) {
      const thresholds = [5, 12, 24, 40, 60, 80];
      nextThreshold = thresholds.find(t => t > this.totalWords) || 'MAX';
    } else if (activePlayers === 3) {
      const thresholds = [6, 15, 30, 50, 75, 100];
      nextThreshold = thresholds.find(t => t > this.totalWords) || 'MAX';
    } else if (activePlayers === 4) {
      const thresholds = [7, 17, 35, 55, 80, 110];
      nextThreshold = thresholds.find(t => t > this.totalWords) || 'MAX';
    } else {
      const thresholds = [8, 20, 40, 65, 95, 130];
      nextThreshold = thresholds.find(t => t > this.totalWords) || 'MAX';
    }

    return nextThreshold === 'MAX' ? 'MAX LEVEL' : `${nextThreshold - this.totalWords} words to next level`;
  }

  clearTimers() {
    if (this.turnTimer) {
      clearTimeout(this.turnTimer);
      this.turnTimer = null;
    }
    if (this.joinTimer) {
      clearInterval(this.joinTimer);
      this.joinTimer = null;
    }
  }
}

function formatPlayerMention(player) {
  return `@${player.username}`;
}

function mentionjid(jid) {
  return '@' + jid.split('@')[0];
}

async function reactToMessage(client, jid, messageId, emoji) {
  try {
    const reactionMessage = {
      react: {
        text: emoji,
        key: {
          remoteJid: jid,
          id: messageId
        }
      }
    };

    client.sendMessage(jid, reactionMessage).catch(console.error);
  } catch (error) {
    console.error('Error reacting to message:', error);
  }
}

function getGameStatusMessage(game) {
  const activePlayers = game.getActivePlayers();
  if (activePlayers.length <= 1) {

    const winner = activePlayers[0];
    let message = winner 
      ? `${mentionjid(winner.id)} Won 🏆\n`
      : `Game Over! No winner 😔\n`;

    message += `Words : *${game.totalWords}*\n`;
    message += `Final Level : ${game.getDifficultyEmoji()} *${game.currentDifficulty.level}*\n`;
    if (game.longestWord) {
      message += `Longest word : *${game.longestWord} (${game.longestWord.length})* by ${game.longestWordPlayer} 📚\n`;
    }
    message += `Time : *${game.getGameDuration()}* ⏱️`;
    return { text: message, mentions: winner ? [winner.id, game.longestWordPlayer] : [game.longestWordPlayer] };
  }

  const currentPlayer = game.getCurrentPlayer();
  const nextPlayer = game.getNextPlayer();

  if (!currentPlayer || !nextPlayer) {
    return { text: "❌ Game error: No valid players found", mentions: [] };
  }

  const statusMessage = `🎲Turn : ${mentionjid(currentPlayer.id)}\n` +
         `🙌Next : ${mentionjid(nextPlayer.id)}\n` +
         `🆎Your word must start with *${game.currentLetter}* and include at least *${game.currentDifficulty.minLength}* letters\n` +
         `${game.getDifficultyEmoji()} Level : *${game.currentDifficulty.level}* (${game.getDifficultyProgressInfo()})\n` +
         `🏆Players remaining : ${activePlayers.length}/${game.players.length}\n` +
         `⏳You have *${game.currentDifficulty.timeLimit/1000}s* to answer\n` +
         `📝Total words : ${game.totalWords}`;

  return { text: statusMessage, mentions: [currentPlayer.id, nextPlayer.id] };
}

function startTurnTimer(game, client) {

  if (game.turnTimer) {
    clearTimeout(game.turnTimer);
  }

  const currentTurnNumber = game.turnNumber;
  const currentPlayer = game.getCurrentPlayer();

  if (!currentPlayer) {
    console.log('No current player found, ending game');
    return;
  }

  game.turnTimer = setTimeout(async () => {

    if (game.turnNumber !== currentTurnNumber) {
      console.log(`Timer expired for old turn ${currentTurnNumber}, current turn is ${game.turnNumber}`);
      return; 
    }

    if (game.processingWord) {
      console.log('Word is being processed, delaying elimination');

      setTimeout(() => {
        if (game.turnNumber === currentTurnNumber && !game.processingWord) {
          handleTimeOut();
        }
      }, 100);
      return;
    }

    async function handleTimeOut() {

      const stillCurrentPlayer = game.getCurrentPlayer();
      if (!stillCurrentPlayer || stillCurrentPlayer.id !== currentPlayer.id) {
        console.log('Current player changed, skipping timeout');
        return;
      }

      console.log(`Player ${currentPlayer.username} timed out`);

      await client.sendMessage(game.jid, {
        text: `${mentionjid(currentPlayer.id)} ran out of time! They're out! 🚫`,
        mentions: [currentPlayer.id]
      });

      game.eliminatePlayer(currentPlayer.id);

      const activePlayers = game.getActivePlayers();
      if (activePlayers.length <= 1) {

        const statusMessage = getGameStatusMessage(game);
        await client.sendMessage(game.jid, statusMessage);
        activeGames.delete(game.jid);
        game.clearTimers();
      } else {

        game.nextTurn(); 
        const statusMessage = getGameStatusMessage(game);
        await client.sendMessage(game.jid, statusMessage);
        startTurnTimer(game, client);
      }
    }

    await handleTimeOut();
  }, game.currentDifficulty.timeLimit);
}

Module({
  pattern: 'wcg ?(.*)',
  fromMe: true,
  desc: 'Start Word Chain Game with Dynamic Difficulty',
  type: 'game'
}, async (message, match) => {
  const jid = message.jid;
  const args = match[1]?.trim().split(' ') || [];
  const command = args[0]?.toLowerCase();

  if (command === 'start') {

    const game = activeGames.get(jid);
    if (!game) {
      return await message.sendReply('❌ No game to start! Use `.wcg` to create a game first.');
    }

    if (game.gameState !== 'waiting') {
      return await message.sendReply('❌ Game is already running or finished!');
    }

    if (game.players.length < 2) {
      return await message.sendReply('❌ Need at least 2 players to start the game!');
    }

    game.clearTimers();

    game.gameState = 'active';
    game.startTime = Date.now();
    game.currentTurnStartTime = Date.now();
    game.turnNumber = 1;

    game.updateDifficulty();

    const statusMessage = getGameStatusMessage(game);
    await message.client.sendMessage(jid, statusMessage);

    startTurnTimer(game, message.client);
    return;
  }

  if (activeGames.has(jid)) {
    return await message.sendReply('❌ A game is already active in this chat!');
  }

  const game = new WordChainGame(jid);
  activeGames.set(jid, game);

  let gameMessage = `🎮 Dynamic Word Chain Game Starting...\n` +
                   `👥 Needs 2 or more players 🙋‍♂️🙋‍♀️\n` +
                   `⏳ Type *join*, 60 seconds left to join ⏳\n` +
                   `🔄 *Difficulty increases automatically as you play!*\n` +
                   `🟢 Start: 3+ letters, 40s per turn\n` +
                   `⚫ Master: 6+ letters, 15s per turn\n` +
                   `⚡ *Fewer players = Faster level progression!*\n` +
                   `📊 *2 players: Every 5-12-24... words*\n` +
                   `📊 *3 players: Every 6-15-30... words*\n` +
                   `📊 *4+ players: Every 8-20-40... words*`;

  await message.sendReply(gameMessage);

  let countdown = 60;
  game.joinTimer = setInterval(async () => {
    countdown -= 15;
    if (countdown > 0) {
      let updateMessage = `🎮 Game starts in ${countdown} seconds ⏳\n` +
                         `Type *join* to play 🙋‍♂️🙋‍♀️\n` +
                         `🔄 Dynamic difficulty system\n` +
                         `⚡ Fewer players = Faster progression`;

      if (game.players.length > 0) {
        updateMessage += `\n👥 ${game.players.length} players joined.`;
      }

      await message.client.sendMessage(jid, { text: updateMessage });
    } else {
      clearInterval(game.joinTimer);
      game.joinTimer = null;

      if (game.players.length >= 2) {
        game.gameState = 'active';
        game.startTime = Date.now();
        game.currentTurnStartTime = Date.now();
        game.turnNumber = 1;

        game.updateDifficulty();

        const statusMessage = getGameStatusMessage(game);
        await message.client.sendMessage(jid, statusMessage);

        startTurnTimer(game, message.client);
      } else {
        await message.client.sendMessage(jid, { 
          text: '❌ Not enough players joined! Game cancelled.' 
        });
        activeGames.delete(jid);
      }
    }
  }, 15000);
});

Module({
  on: 'text',
  fromMe: false
}, async (message) => {
  const jid = message.jid;
  const game = activeGames.get(jid);

  if (!game) return;
  const text = message.message?.trim()?.toLowerCase();
  const userId = message.sender;
  const username = message.pushName || message.data.pushName || userId?.split('@')?.[0] || 'Unknown';

  if (text === 'join' && game.gameState === 'waiting') {
    if (game.addPlayer(userId, username)) {
      await message.client.sendMessage(jid, {
        text: `${mentionjid(userId)} Joined 👏`,
        mentions: [userId]
      });
    } else {
      await message.client.sendMessage(jid, {
        text: `${mentionjid(userId)} You're already in the game!`,
        mentions: [userId]
      });
    }
    return;
  }

  if (game.gameState === 'active') {
    const currentPlayer = game.getCurrentPlayer();

    if (!currentPlayer || currentPlayer.id !== userId) {
      return; 
    }

    if (game.processingWord) {
      return; 
    }

    game.processingWord = true;

    try {
      const word = text.trim();

      if (!/^[a-zA-Z]+$/.test(word)) {
        return; 
      }

      if (word.charAt(0).toUpperCase() !== game.currentLetter) {
        reactToMessage(message.client, jid, message.data.key.id, '❌');
        await message.sendReply(`❌ Word must start with *${game.currentLetter}*!`);
        return;
      }

      if (word.length < game.currentDifficulty.minLength) {
        reactToMessage(message.client, jid, message.data.key.id, '❌');
        await message.sendReply(`❌ Word must be at least *${game.currentDifficulty.minLength}* letters long! (${game.currentDifficulty.level} level)`);
        return;
      }

      if (game.usedWords.has(word.toLowerCase())) {
        reactToMessage(message.client, jid, message.data.key.id, '🔄');
        await message.sendReply(`❌ Word *${word}* has already been used!`);
        return;
      }

      if (!game.isValidWord(word)) {
        reactToMessage(message.client, jid, message.data.key.id, '❓');
        await message.sendReply('_This word is not in my words list_');
        return;
      }

      if (game.turnTimer) {
        clearTimeout(game.turnTimer);
        game.turnTimer = null;
      }

      reactToMessage(message.client, jid, message.data.key.id, '✅');

      game.usedWords.add(word.toLowerCase());
      game.totalWords++;
      currentPlayer.words++;

      if (word.length > game.longestWord.length) {
        game.longestWord = word.toLowerCase();
        game.longestWordPlayer = mentionjid(currentPlayer.id);

        reactToMessage(message.client, jid, message.data.key.id, '🏆');
      }

      const levelChanged = game.updateDifficulty();

      game.currentLetter = word.charAt(word.length - 1).toUpperCase();

      game.nextTurn();

      if (levelChanged) {
        await message.client.sendMessage(jid, {
          text: `🆙 *LEVEL UP!* ${game.getDifficultyEmoji()}\n` +
                `New Level: *${game.currentDifficulty.level}*\n` +
                `Min Length: *${game.currentDifficulty.minLength}* letters\n` +
                `Time Limit: *${game.currentDifficulty.timeLimit/1000}s* per turn\n` +
                `Progress: ${game.getDifficultyProgressInfo()}`
        });
      }

      const statusMessage = getGameStatusMessage(game);
      await message.client.sendMessage(jid, statusMessage);

      startTurnTimer(game, message.client);

    } finally {

      game.processingWord = false;
    }
  }
});

Module({
  pattern: 'endwcg',
  fromMe: true,
  desc: 'End current Word Chain Game',
  type: 'game'
}, async (message) => {
  const jid = message.jid;
  const game = activeGames.get(jid);

  if (!game) {
    return await message.sendReply('❌ No active game in this chat!');
  }

  game.clearTimers();

  activeGames.delete(jid);

  await message.sendReply('🛑 Word Chain Game ended by admin!');
});

Module({
  pattern: 'wcgstats',
  fromMe: true,
  desc: 'Show current Word Chain Game stats',
  type: 'game'
}, async (message) => {
  const jid = message.jid;
  const game = activeGames.get(jid);

  if (!game) {
    return await message.sendReply('❌ No active game in this chat!');
  }

  let statsMessage = `📊 *Word Chain Game Stats*\n\n`;
  statsMessage += `🎮 State: ${game.gameState}\n`;
  statsMessage += `${game.getDifficultyEmoji()} Level: ${game.currentDifficulty.level}\n`;
  statsMessage += `📏 Min Length: ${game.currentDifficulty.minLength} letters\n`;
  statsMessage += `⏱️ Time Limit: ${game.currentDifficulty.timeLimit/1000}s\n`;
  statsMessage += `👥 Total Players: ${game.players.length}\n`;
  statsMessage += `🏆 Active Players: ${game.getActivePlayers().length}\n`;
  statsMessage += `📝 Words used: ${game.totalWords}\n`;
  statsMessage += `🎯 Progress: ${game.getDifficultyProgressInfo()}\n`;

  if (game.longestWord) {
    statsMessage += `📏 Longest word: ${game.longestWord} (${game.longestWord.length})\n`;
  }

  if (game.gameState === 'active') {
    statsMessage += `🔤 Current letter: ${game.currentLetter}\n`;
    statsMessage += `⏱️ Duration: ${game.getGameDuration()}\n`;
    statsMessage += `🎯 Turn: ${game.turnNumber}\n`;
  }

  await message.sendReply(statsMessage);

});

/*
 * Game Logic: Word Chain Game with Dynamic Difficulty
 * Written by @souravkl11
 * Date: 2025-06-15
 *
 * This module implements a word chain game featuring:
 * - Dynamic difficulty adjustment based on active players and total words played.
 * - Turn-based gameplay with time limits for each player.
 * - Player elimination for incorrect words or timeouts.
 *
 * Word list adapted from 'dwyl/english-words' project.
 * Original Source: https://rawcdn.githack.com/dwyl/english-words/20f5cc9b3f0ccc8ce45d814c532b7c2031bba31c/words_alpha.txt
 *
 * License: GPL 3.0
 */